package easik.states;

import java.util.ArrayList;

import javax.swing.JOptionPane;

import easik.Easik;
import easik.sketch.Sketch;
import easik.sketch.constraint.Constraint;
import easik.sketch.constraint.PullbackConstraint;
import easik.sketch.path.SketchPath;


/** 
 * This state makes a pullback constraint. When pushed, it will then push another
 * state to get a path. Once the first path is received, it will push another request 
 * for a path. This repeated until four paths have been collected and then the 
 * constraint is created (as long as the paths are valid).
 *
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @since 2006-05-23 Kevin Green
 * @version 2006-08-15 Kevin Green
 */
public class AddPullbackConstraintState extends EasikState implements PathAcceptingState {
	/**
	 * The paths involved in this constraint
	 */
	private ArrayList<SketchPath> _paths;
	/**
	 * Stores whether the user has finished adding paths to this constraint
	 */
	private boolean _finished;
	/**
	 * The current path number being selected
	 */
	private int _round;
	
	/**
	 * Constructor for creating a new state to collect paths and make a diagram.
	 * 
	 * @param inSketch The sketch being worked with.
	 */
	public AddPullbackConstraintState(Sketch inSketch) {
		_ourSketch = inSketch;
		_finished = false;
		_paths = new ArrayList<SketchPath>();
		_round = 0;
	}

	/**
	 * Called when a new path has been completed by a state above this on the
	 * stack. If it is null, then cancel, otherwise add it to the collection.
	 * When we get four paths, then we're finished (if they share a domain and codomain).
	 * 
	 * @param inPath The last path in, null if it was a cancellation
	 */
	public void passPath(SketchPath inPath) {
		_finished = Easik.getInstance().getStateManager().getFinished();
		//Check if cancelled
		if (inPath == null) {
			Easik.getInstance().getStateManager().popState();
			return;
		}
		
		if(!_finished){
			_paths.add( inPath);
			_round++;
			if(_round < 3)
				Easik.getInstance().getStateManager().pushState(new GetPathState(true, false));
			else
				Easik.getInstance().getStateManager().pushState(new GetPathState(false, true));
		}
		else{
			_paths.add( inPath);
			if(addDiagram()){
				JOptionPane.showMessageDialog(
					_ourSketch.getParent(),
					"Created a Pullback Constraint",
					"JEase",
					JOptionPane.INFORMATION_MESSAGE);
				
			}
			else{
				JOptionPane.showMessageDialog(_ourSketch.getParent(), 
						"Invalid path configuration.\n"+
						"Please ensure that:\n"+
						"1) Paths are selected in the order previously described.\n"+
						"2) Injective edges form a valid configuration.\n"+
						"Pullback constraint not created",
						"Error",
						JOptionPane.ERROR_MESSAGE);
			}
			Easik.getInstance().getStateManager().popState();
		}

	}

	/**
	 * Add the diagram to the sketch
	 *
	 * @return true if the constraint was successfully added to the sketch, false otherwise
	 */
	private boolean addDiagram() {
		if(PullbackConstraint.isPullbackConstraint(_paths)){
			Constraint newDiagram = new PullbackConstraint(_paths);
			Easik.getInstance().getFrame().getSketch().addNewConstraint(newDiagram);
			Easik.getInstance().getFrame().getSketch().setDirty(true);
			return true;
		}
		return false;
	}


	/**
	 * When this state is pushed on, it sends a message in a popup and then pushes
	 * a path collection state.
	 */
	public void pushedOn() {
		Easik.getInstance().getFrame().getSketch().getSelectionListener().emptySelection();
		Easik.getInstance().getStateManager().resetFinished();
		JOptionPane.showMessageDialog(
			_ourSketch.getParent(),
			"Select a path and then press 'Next' to add it to the constraint and 'Finish' when you are done adding paths",
			"JEase",
			JOptionPane.INFORMATION_MESSAGE);

		Easik.getInstance().getFrame().setStateString("Pullback Constraint");
		Easik.getInstance().getFrame().getNextButton().setEnabled(true);
		Easik.getInstance().getFrame().getCancelButton().setEnabled(true);
		Easik.getInstance().getStateManager().pushState(new GetPathState(true, false));
	}

	/**
	 * Nothing to do when it gets popped off, as it always pops itself off when completed. 
	 */
	public void poppedOff() {
	}

	/**
	 * This state is called "New Pullback Constraint"
	 * 
	 * @return The string literal described above.
	 */
	public String toString() {
		return "New Pullback Constraint";
	}

}

